/*
 * Copyright (c) 2008-2016, RF-Embedded GmbH
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "QrfeFifo.h"

#include <QTimer>


/**
 * @brief QrfeFifo::QrfeFifo constructs a new object
 * @param parent    Parent object
 */
QrfeFifo::QrfeFifo(QObject *parent) :
    QIODevice(parent)
{
	m_mutex = new QMutex(QMutex::Recursive);
}

/**
 * @brief QrfeFifo::~QrfeFifo Destructor
 */
QrfeFifo::~QrfeFifo()
{
	delete m_mutex;
}

/**
 * @brief QrfeFifo::open opens the fifo
 * @param mode      The open mode
 * @return Success of the operation
 */
bool QrfeFifo::open ( OpenMode mode )
{
    return QIODevice::open(mode | QIODevice::Unbuffered);
}

/**
 * @brief QrfeFifo::reset resets the fifo
 * @return Success of the operation
 */
bool QrfeFifo::reset ()
{
    QMutexLocker lock(m_mutex);
    m_buffer.clear();

    return true;
}

/**
 * @brief QrfeFifo::bytesAvailable indicates if bytes to read are available
 * @return The count of bytes to read
 */
qint64 QrfeFifo::bytesAvailable () const
{
	QMutexLocker lock(m_mutex);
	qint64 ret = m_buffer.size();

    return ret;
}

/**
 * @brief QrfeFifo::readData reads data out of the fifo. The function is used by the functions of QIODevice.
 * @param data      Buffer to store data in
 * @param maxSize   The maximum size of the buffer
 * @return The count of bytes read
 */
qint64 QrfeFifo::readData ( char * data, qint64 maxSize )
{
	QMutexLocker lock(m_mutex);
	qint64 size = qMin(maxSize, (qint64)m_buffer.size());
	if(size > 0)
	{
		memcpy(data, m_buffer.data(), size);
		m_buffer.remove(0, size);
	}

	return size;
}

/**
 * @brief QrfeFifo::writeData writes data to the fifo. The function is used by the functions of QIODevice.
 * @param data      Pointer to the bytes that should be written.
 * @param maxSize   The size of the buffer
 * @return The count of bytes written
 */
qint64 QrfeFifo::writeData ( const char * data, qint64 maxSize )
{
	m_mutex->lock();;
	m_buffer.append(data, maxSize);
	m_mutex->unlock();

	//emit readyRead();
	QMetaObject::invokeMethod(this, "_q_emitSignals", Qt::QueuedConnection);

	return maxSize;
}

/**
 * @brief QrfeFifo::_q_emitSignals is a little helper to emit readyRead queued.
 */
void QrfeFifo::_q_emitSignals()
{
	emit readyRead();
}
